#include <stdlib.h>
#include <memory.h>
#include <openssl/bio.h>
#include <openssl/rsa.h>
#include <openssl/pem.h>
#include <openssl/sha.h>
#include <openssl/err.h>

#define OK			1
#define KO			0
#define LG_DATE_HEURE	12
#define LG_SIGN	256
#define LG_BUFFER	512
#define LG_CLE_PUBLIQUE 392

#define CLE_PUBLIQUE_DGFIP_PEDI "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA5JNDEX4rKlgoOhHFYIvf77Eyw0XpC4VwSUFZKic/ggnBTZWUjtU0dw5fcIWiDc4WMML5NXzDfz0rPcpiPz+ly+CmLibgemUk/FfYGd0S/Wxj3t3Qfc2Tv1yC1vBmdXvNenv9u9jUQvuW3NUgWMz/oh/PnWC/fSaYABO1T7XLxEDIPbEni9brx/ZLH5EUpzm/rz56J0If6BfOAqYC39d6HFwg/d6SjwVtKsgOx2zyRVEo9dd8tjeYR7sW/ZbsOHvkb4yHzcfpn6H2hn1/SABjQNViZVKdG0fZsHzXh6SGoOuYCicRdojCd29PoBqV4B6ISrH+lZ1Ydgg62B5Y+xoBSQIDAQAB"

/*
 * La seule vocation de ce programme est de vrifier la signature d'un fichier transmis en retour par la DGFiP (message CONTRL, INFENT CR...)
 * 
 * Un paramtre en entre du programme :
 *
 *    - le fichier sign  contrler
 *
 * En sortie, le programme fournit le rsultat de la vrification
 *
 */

int main ( int argc, char * * argv)
{
	RSA * clePublique ;
	FILE * ptrFichier ;
	BIO * ptrClePublique ;
	
	char fichierGF [ LG_BUFFER ] ;
	
	SHA256_CTX contexteSHA256 ;
	unsigned char hash [ SHA256_DIGEST_LENGTH + LG_DATE_HEURE + 1 ] ;
	unsigned char signDecrypt [ LG_BUFFER ] ;
	
	unsigned char buffer_lecture [ LG_BUFFER ] ;
	unsigned char signature_bin [ LG_SIGN + 1 ] ;
	unsigned char signature_hexa [ LG_SIGN * 2 + 1 ] ;
	unsigned char date_heure [ LG_DATE_HEURE + 1 ] ;
	unsigned char cle_publique_hexa [ LG_CLE_PUBLIQUE * 2 + 1 ] ;
	unsigned char cle_publique_bin [ LG_CLE_PUBLIQUE + 1 ] ;
	unsigned char cle_publique_pem [ LG_BUFFER ] ;

	int compteur ;
	int nb_car_lus ;
	int caractere ;
	int	mode_signature ;
	
	if ( argc != 2 )
	{
		fprintf ( stderr, "Usage : %s <fichier_a_controler>\n", argv[0] ) ;
		
		exit ( 1 ) ;
	}
	
	/* Rcupration des informations de signature                                                */
	/* ------------------------------------------                                                */
	/* A partir du fichier  contrler :                                                         */ 
	/* - cration d'un fichier temporaire contenant uniquement le groupe fonctionnel principal ; */
	/* - rcupration de la date et de l'heure de signature ;                                    */
	/* - rcupration de la signature en hexadcimal ;                                           */
	/* - rcupration de la cl publique DGFiP "partenaire EDI" en hexadcimal                   */
	/* ========================================================================================= */
	sprintf ( fichierGF, "/tmp/%s_%d", basename(argv[0]), getpid()) ;
	memset ( date_heure, 0, sizeof ( date_heure ) ) ;
	memset ( signature_hexa, 0, sizeof ( signature_hexa ) ) ;
	memset ( cle_publique_hexa, 0, sizeof ( cle_publique_hexa ) ) ;

	if ( recuperer_elements_signature ( argv[1], fichierGF, date_heure, signature_hexa, cle_publique_hexa )  != OK )
	{
		exit ( 1 ) ;
	}
	
	/* Conversion de la cl publique au format hexadcimal en une cl au format binaire (texte) */
	/* ======================================================================================== */
	memset ( cle_publique_bin, 0, sizeof ( cle_publique_bin ) ) ;
	for ( compteur = 0 ; compteur < (LG_CLE_PUBLIQUE * 2) ; compteur += 2 )
    	{
      		if ( sscanf ( cle_publique_hexa + compteur, "%2x", & caractere ) == 0 )
      		{
          		fprintf ( stderr, "Erreur de conversion de la signature d'hexadcimal en binaire\n" ) ;

			unlink ( fichierGF ) ;
		  
		  	exit ( 1 ) ;
      		}
	  	cle_publique_bin [compteur / 2] =  (unsigned char) caractere ;
    	}
	
	/* On vrifie que la cl publique contenue dans l'accrditation est bien celle de la DGFiP */
	/* (contrle facultatif puisque cette cl publique est unique et donc toujours la mme...) */
	/* ======================================================================================= */
	if ( strcmp ( cle_publique_bin, CLE_PUBLIQUE_DGFIP_PEDI ) )
	{
          	fprintf ( stderr, "La cl publique lue dans l'accrditation ne correspond pas  la cl publique DGFiP \"partenaire EDI\".\n" ) ;
		  
		fprintf ( stderr, "Cl publique lue      => %s\n", cle_publique_bin ) ;
		fprintf ( stderr, "Cl publique attendue => %s\n", CLE_PUBLIQUE_DGFIP_PEDI ) ;

		unlink ( fichierGF ) ;
		  
		exit ( 1 ) ;
	}
	
	/* Construction de la cl publique au format PEM */
	/* ============================================= */
	sprintf ( cle_publique_pem, "-----BEGIN PUBLIC KEY-----" ) ;
	for ( compteur = 0 ; compteur < LG_CLE_PUBLIQUE ; compteur++ )
	{
		if ( (compteur % 64) == 0 )
		{
			strcat ( cle_publique_pem, "\n" ) ;
		}
		strncat ( cle_publique_pem, cle_publique_bin + compteur, 1 ) ;
	}
	strcat ( cle_publique_pem, "\n-----END PUBLIC KEY-----" ) ;

	/* Initialisation */
	/* ============== */
	OpenSSL_add_all_ciphers() ;
	memset ( hash, 0, sizeof ( hash ) ) ;
	
	/* Lecture de la cl publique */
	/* ========================== */
	ptrClePublique = BIO_new_mem_buf ( cle_publique_pem, -1 ) ;
	clePublique = (RSA *) PEM_read_bio_RSA_PUBKEY ( ptrClePublique, NULL, 0, NULL ) ;
	BIO_set_close ( ptrClePublique, BIO_CLOSE ) ;
	BIO_free ( ptrClePublique ) ;

	if ( clePublique == NULL )
	{
		fprintf ( stderr, "Erreur de lecture de la cle publique\n" ) ;
		ERR_print_errors_fp ( stderr ) ;
		unlink ( fichierGF ) ;

		exit ( 1 ) ;
	}
	
	/* Cration de l'empreinte horodate */
	/* =================================
	
	/* Chargement des donnes  "hasher" dans le contexte SHA */
	/* ------------------------------------------------------ */
	
	/* Initialisation */
	if ( SHA256_Init ( & contexteSHA256 ) != OK )
	{
		fprintf ( stderr, "Erreur d'initialisation du contexte pour le calcul de l'empreinte SHA256\n" ) ;
		ERR_print_errors_fp ( stderr ) ;
		unlink ( fichierGF ) ;

		exit ( 1 ) ;
	}
	ptrFichier = fopen ( fichierGF, "r" ) ;

	if ( ptrFichier == NULL )
	{
		fprintf ( stderr, "Erreur d'ouvertue du fichier %s contenant le groupe fonctionnel extrait du fichier  contrler\n", fichierGF ) ;
		unlink ( fichierGF ) ;
		exit ( 1 ) ;
	}
	/* Chargement des donnes */
	while ( ( nb_car_lus = fread ( buffer_lecture, 1, LG_BUFFER, ptrFichier ) ) )
	{
		if ( SHA256_Update ( & contexteSHA256, buffer_lecture, nb_car_lus ) != OK )
		{
			fclose ( ptrFichier ) ;
			fprintf ( stderr, "Erreur de mise  jour du contexte pour le calcul de l'empreinte SHA256\n" ) ;
			ERR_print_errors_fp ( stderr ) ;
			unlink ( fichierGF ) ;

			exit ( 1 ) ;
		}
	}
	
	if ( ferror ( ptrFichier ) )
	{
		fprintf ( stderr, "Erreur de lecture du fichier %s contenant le groupe fonctionnel extrait du fichier  contrler\n", fichierGF ) ;
		unlink ( fichierGF ) ;

		exit ( 1 ) ;
	}
	fclose ( ptrFichier ) ;
	
	/* Suppression du fichier temporaire contenant le groupe fonctionnel extrait */
	unlink ( fichierGF ) ;
	
	/* Calcul de l'empreinte */
	if ( SHA256_Final ( hash, & contexteSHA256 ) != OK )
	{
		fprintf ( stderr, "Erreur de calcul de l'empreinte SHA1\n" ) ;
		ERR_print_errors_fp ( stderr ) ;

		exit ( 1 ) ;
	}
	
	/* Ajout de la date (AAAAMMJJ) et de l'heure (HHMM)  */
	if ( strlen ( date_heure ) != LG_DATE_HEURE )
	{
		fprintf ( stderr, "La date et l'heure lues ne sont pas au format AAAAMMJJhhmm\n" ) ;
		
		exit ( 1 ) ;
	}
	memcpy ( hash + SHA256_DIGEST_LENGTH, date_heure, LG_DATE_HEURE ) ;

	/* Dchiffrement de la signature */
	/* ============================= */
	/* (on commence par la convertir d'hexa en binaire) */
	memset ( signature_bin, 0, sizeof ( signature_bin ) ) ;
	for ( compteur = 0 ; compteur < (LG_SIGN * 2) ; compteur += 2 )
    	{
      		if ( sscanf ( signature_hexa + compteur, "%2x", & caractere ) == 0 )
      		{	
          		fprintf ( stderr, "Erreur de conversion de la signature d'hexadcimal en binaire\n" ) ;
		  
		  	exit ( 1 ) ;
      		}
	  	signature_bin [compteur / 2] =  (unsigned char) caractere ;
    	}	
	
	memset ( signDecrypt, 0, sizeof ( signDecrypt ) ) ;

	if ( RSA_public_decrypt ( LG_SIGN, signature_bin, signDecrypt, clePublique, RSA_PKCS1_PADDING ) == -1 )
	{
		fprintf ( stderr, "Erreur de dchiffrement de la signature\n" ) ;
		ERR_print_errors_fp ( stderr ) ;

		exit ( 1 ) ;
	}
   
	RSA_free ( clePublique ) ;

	/* Comparaison entre l'empreinte horodate calcule et la signature dchiffre */
	/* =========================================================================== */
	if ( memcmp ( hash, signDecrypt, SHA_DIGEST_LENGTH + LG_DATE_HEURE ) )
	{
		fprintf ( stdout, "Signature KO !\n" ) ;

		fprintf ( stderr, "Hash calcul        => " ) ;
		for ( compteur = 0 ; compteur < SHA_DIGEST_LENGTH + LG_DATE_HEURE ; compteur++ )
		{
			fprintf ( stderr, "%02X", hash[compteur] ) ;
		}
		fprintf ( stderr, "\nSignature dcrypte => " ) ;
		for ( compteur = 0 ; compteur < SHA_DIGEST_LENGTH + LG_DATE_HEURE ; compteur++ )
		{
			fprintf ( stderr, "%02X", signDecrypt[compteur] ) ;
		}
		fprintf ( stderr, "\n" ) ;
	}
	else
	{
		fprintf ( stdout, "Signature OK !\n" ) ;
	}
	
	return ( 0 ) ;
}

/*
 *	- Extraction du groupe fonctionnel principal (le premier aprs l'UNB) dans le fichier,
 *	  avec cration d'un fichier temporaire dont le chemin est stock dans la variable fournie en paramtre
 *	- Rcupration de la date et de l'heure de signature (retournes dans la variable fournie en paramtre
 *	  au format AAAAMMJJHHMM)
 *	- Rcupration de la signature en hexadcimal
 *	- Rcupration de la cl publique contenue dans l'accrditation, en hexadcimal
 *	
 *	Renvoie OK ou KO
 */

int recuperer_elements_signature ( char * fichierAControler, char * fichierTmp, char * date_heure, char * signature, char * cle_publique )
{
	int   vl_iI, vl_iPos_Deb, vl_iPos_Fin, caractere ;
	FILE   * vl_psInter_Trav, * vl_psFich_Extrait ;

	/* Ouverture du fichier (interchange) */
	/* ================================== */
	vl_psInter_Trav = fopen ( fichierAControler, "r" ) ;
	if ( vl_psInter_Trav == NULL )
	{
		fprintf ( stderr, "Erreur d'ouverture du fichier %s\n", fichierAControler ) ;
		return ( KO ) ;
	}

	/* Recherche de la squence correspondant au */
	/* dbut du groupe fonctionnel des donnes   */
	/* ========================================= */
	if ( trouver_chaine_dans_fichier ( vl_psInter_Trav, "'UNG+", & vl_iPos_Deb ) == KO )
	{
		fprintf ( stderr, "Erreur : premier segment UNG (dbut des donnes signes) non trouv\n" ) ;
		fclose ( vl_psInter_Trav ) ;

		return ( KO ) ;
	}

	/* On se repositionne dans le fichier... */
	/* ------------------------------------- */
	fseek ( vl_psInter_Trav, vl_iPos_Deb + 1, SEEK_SET ) ;

	/* Recherche de la squence correspondant  */
	/* la fin du groupe fonctionnel des donnes */
	/* ======================================== */
	if ( trouver_chaine_dans_fichier ( vl_psInter_Trav, "'UNE+", & vl_iPos_Fin ) == KO )
	{
		fprintf ( stderr, "Erreur : premier segment UNE (fin des donnes signes) non trouv\n" ) ;
		fclose ( vl_psInter_Trav ) ;

		return ( KO ) ;
	}

	/* Recherche de la fin du groupe fonctionnel courant */
	/* ================================================= */
	if ( trouver_chaine_dans_fichier ( vl_psInter_Trav, "'", & vl_iPos_Fin ) == KO )
	{
		fprintf ( stderr, "Erreur : fin du premier segment UNE (donnes signes) non trouve\n" ) ;
		fclose ( vl_psInter_Trav ) ;

		return ( KO ) ;
	}

	/* Cration du fichier issu de l'extraction des donnes signes */
	/* ============================================================ */

	/* Ouverture du fichier */
	/* -------------------- */
	vl_psFich_Extrait = fopen ( fichierTmp, "w" ) ;

	if ( vl_psFich_Extrait == NULL )
	{
		fprintf ( stderr, "Erreur de cration du fichier temporaire %s\n", fichierTmp ) ;
		fclose ( vl_psInter_Trav ) ;

		return ( KO ) ;
	}

	/* Repositionnement du pointeur de fichier */
	/* --------------------------------------- */
	if ( fseek ( vl_psInter_Trav, vl_iPos_Deb + 1, SEEK_SET ) )
	{
		fprintf ( stderr, "Erreur de positionnement sur le dbut du premier segment UNG (dbut des donnes signes)\n" ) ;
		fclose ( vl_psInter_Trav ) ;
		fclose ( vl_psFich_Extrait ) ;
		unlink ( fichierTmp ) ;
	
		return ( KO ) ;
	}

	/* Alimentation du fichier contenant l'extraction */
	/* ---------------------------------------------- */
	for ( vl_iI = vl_iPos_Deb ; vl_iI < vl_iPos_Fin ; vl_iI++ )
	{
		/* On vrifie que le caractre est bien crit */
		/* ------------------------------------------ */
		if ( fprintf ( vl_psFich_Extrait, "%c", fgetc ( vl_psInter_Trav ) ) != 1 )
		{
			fprintf ( stderr, "Erreur d'criture dans le fichier temporaire %s\n", fichierTmp ) ;
			fclose ( vl_psInter_Trav ) ;
			fclose ( vl_psFich_Extrait ) ;
			unlink ( fichierTmp ) ;

			return ( KO ) ;
		}
	}
  
	/* Lecture de la date et de l'heure de signature */
	/* --------------------------------------------- */
  
	/* On commence par stocker les deux premiers chiffres de l'anne */
	strcpy ( date_heure, "20" ) ;
  
	/* On se dplace de 27 cacatres pour se positionner sur la date (au format AAMMJJ) */
	if ( fseek ( vl_psInter_Trav, 27, SEEK_CUR ) )
	{
		fprintf ( stderr, "Erreur de positionnement sur le dbut de la date de signature (dans l'AUTACK)\n" ) ;
		fclose ( vl_psInter_Trav ) ;
		fclose ( vl_psFich_Extrait ) ;
		unlink ( fichierTmp ) ;
	
		return ( KO ) ;
	}
 
	/* On lit la date (AAMMJJ) */
	if ( fgets ( (char *)(date_heure + 2), 7 /* 6 caractres  lire plus le zro de fin de chane */, vl_psInter_Trav ) == NULL )
	{
		fprintf ( stderr, "Erreur de lecture de la date de signature (dans l'AUTACK)\n" ) ;
		fclose ( vl_psInter_Trav ) ;
		fclose ( vl_psFich_Extrait ) ;
		unlink ( fichierTmp ) ;

		return ( KO ) ;
	}
  
	/* On laisse de ct le sparateur */
	if ( fseek ( vl_psInter_Trav, 1, SEEK_CUR ) )
	{
		fprintf ( stderr, "Erreur de positionnement sur l'heure de signature (dans l'AUTACK)\n" ) ;

		fclose ( vl_psInter_Trav ) ;
		fclose ( vl_psFich_Extrait ) ;
		unlink ( fichierTmp ) ;

		return ( KO ) ;
	}

	/* On lit l'heure (HHMM) */
	if ( fgets ( (char *)(date_heure + 8), 5 /* 4 caractres  lire plus le zro de fin de chane */, vl_psInter_Trav ) == NULL )
	{
		fprintf ( stderr, "Erreur de lecture de l'heure de signature (dans l'AUTACK)\n" ) ;
		fclose ( vl_psInter_Trav ) ;
		fclose ( vl_psFich_Extrait ) ;
		unlink ( fichierTmp ) ;
	
		return ( KO ) ;
	}
  
	/* Pour rcuprer la cl publique contenue dans l'accrdidation,            */
	/* il faut conserver la chane de longueur 784 et commenant en position 72 */
	/* ======================================================================== */
	if ( trouver_chaine_dans_fichier ( vl_psInter_Trav, "'USR+", & vl_iPos_Deb ) == KO )
	{
		fprintf ( stderr, "Erreur : segment USR (contenant l'accrditation) introuvable\n" ) ;
		fclose ( vl_psInter_Trav ) ;
		fclose ( vl_psFich_Extrait ) ;
		unlink ( fichierTmp ) ;

		return ( KO ) ;
	}
  
	/* Repositionnement du pointeur de fichier              */
	/* (sur le 73me caractre de l'accrd. du segment USR) */
	/* ---------------------------------------------------- */
	if ( fseek ( vl_psInter_Trav, vl_iPos_Deb + 79, SEEK_SET ) )
	{
		fprintf ( stderr, "Erreur de positionnement sur la cl publique contenue dans l'accrditation (segment USR)\n" ) ;
		fclose ( vl_psInter_Trav ) ;
		fclose ( vl_psFich_Extrait ) ;
		unlink ( fichierTmp ) ;

		return ( KO ) ;
	}
  
	/* On lit la cl publique */
	if ( fgets ( cle_publique, (LG_CLE_PUBLIQUE*2 + 1), vl_psInter_Trav ) == NULL )
	{
		fprintf ( stderr, "Erreur de lecture des %d caractres de la cl publique contenue dans l'accrditation (segment USR)\n", LG_CLE_PUBLIQUE*2 ) ;
		fclose ( vl_psInter_Trav ) ;
		fclose ( vl_psFich_Extrait ) ;
		unlink ( fichierTmp ) ;
	
		return ( KO ) ;
	}

	/* Recherche de la squence correspondant au segment contenant la signature */
	/* ======================================================================== */
	if ( trouver_chaine_dans_fichier ( vl_psInter_Trav, "'USY+", & vl_iPos_Deb ) == KO )
	{
		fprintf ( stderr, "Erreur : segment USY (contenant la signature) introuvable aprs lecture des %d caractres de la cl publique contenue dans l'accrditation (segment USR)\n", LG_CLE_PUBLIQUE*2 ) ;
		fclose ( vl_psInter_Trav ) ;
		fclose ( vl_psFich_Extrait ) ;
		unlink ( fichierTmp ) ;
	
		return ( KO ) ;
	}

	/* Repositionnement du pointeur de fichier        */
	/* (au dbut de la signature dans le segment USY) */
	/* ---------------------------------------------- */
	if ( fseek ( vl_psInter_Trav, vl_iPos_Deb + 9, SEEK_SET ) )
	{
		fprintf ( stderr, "Erreur de positionnement sur la signature (segment USY)\n" ) ;
		fclose ( vl_psInter_Trav ) ;
		fclose ( vl_psFich_Extrait ) ;
		unlink ( fichierTmp ) ;

		return ( KO ) ;
	}
  
	/* On lit la signature */
	if ( fgets ( signature, (LG_SIGN*2 + 1), vl_psInter_Trav ) == NULL )
	{
		fprintf ( stderr, "Erreur de lecture des %d caractres de la signature (segment USY)\n", LG_SIGN*2 ) ;
		fclose ( vl_psInter_Trav ) ;
		fclose ( vl_psFich_Extrait ) ;
		unlink ( fichierTmp ) ;

		return ( KO ) ;
	}
  
	/* Fermeture des fichiers */
	/* ====================== */
	fclose ( vl_psInter_Trav ) ;
	fclose ( vl_psFich_Extrait ) ;

	return ( OK ) ;
}

int trouver_chaine_dans_fichier (FILE * vl_psFichID, char * vl_pcChaine, int * vl_piPos )
{
	char  * vl_pcPtr ;   /* Pointeur sur un caractre de la chane */
	int   vl_iCarPrec, vl_iChar ;   

	/* Initialisations */
	/*=================*/
	vl_pcPtr = vl_pcChaine ;
	vl_iCarPrec = -99999 ;

	if ( vl_psFichID == NULL )
		return ( KO ) ;

	while ((vl_iChar = fgetc (vl_psFichID)) != EOF)
	{
		/* Si on lit le premier caractre de la chane le       */
		/* car. prcdent ne doit pas tre le car. d'chap. '?' */
		/* ---------------------------------------------------- */
		if ( vl_pcPtr == vl_pcChaine && vl_iCarPrec == '?' )
		{
			vl_iCarPrec = vl_iChar ;
			continue ;
		}

		if (vl_iChar != * vl_pcPtr)
		{
			if (vl_pcPtr != vl_pcChaine)
			{
			fseek (vl_psFichID, * vl_piPos, SEEK_SET) ;
			vl_pcPtr = vl_pcChaine ;
			}
			else /* Mmorisation du caractre prcdent */
				vl_iCarPrec = vl_iChar ;
			continue ;
		}

		if (vl_pcPtr == vl_pcChaine)
		{
			* vl_piPos = ftell (vl_psFichID) ;
			vl_iCarPrec = vl_iChar ;
		}

		vl_pcPtr ++ ;

		if (! * vl_pcPtr)
		{
			/* Le dbut rel de la chane est position-1 */
			/* ----------------------------------------- */
			( * vl_piPos ) -- ;
			return ( OK ) ;
		}
	}

	return ( KO ) ;
}
